package edu.uky.ai.lp.ai;

import java.util.HashSet;
import java.util.Set;

import edu.uky.ai.lp.Action;
import edu.uky.ai.lp.Game;
import edu.uky.ai.lp.Result;
import edu.uky.ai.lp.Settings;
import edu.uky.ai.lp.logic.*;

/**
 * An artificially intelligent agent which uses logical reasoning based on
 * observations to decide which squares are safe to visit.
 * 
 * @author Your Name
 */
public class LogicAgent implements Agent {

	/** All the rules and facts known about the Wumpus World */
	private final KnowledgeBase kb = WumpusWorld.getKnowledgeBase();
	
	/** The player's current location */
	private String location = "a1";
	
	/**
	 * The destination that the player should move toward (when different from
	 * {@link #location}), the player will move through visited squares to
	 * reach this square)  */
	private String destination = location;
	
	/** Records the location of the gold once it is found */
	private String gold = null;
	
	/** A set of visited squares */
	private Set<String> visited = new HashSet<String>();
	
	/**
	 * Creates a new logical agent.
	 */
	public LogicAgent() {
		// Print the knowledge base.
		System.out.println("KNOWLEDGE BASE:\n" + kb + "\n");
		// Mark the starting location as visited.
		visited.add(location);
	}
	
	@Override
	public Action chooseAction(Game game) {
		Action action = null;
		// If location != destination, move toward destination.
		if(!location.equals(destination)) {
			action = Path.path(location, destination, visited);
			location = Path.getNewLocation(location, action);
		}
		// If the gold is in the current location, pick it up and move to the
		// starting square.
		else if(location.equals(gold)) {
			destination = "a1";
			action = Action.GRAB;
		}
		// Otherwise, use logical deduction to find a square which is safe and
		// unvisited via the 'hint' predicate.
		else {
			Variable x = new Variable("X");                     // Declare a variable X.
			Fact query = new Fact("hint", x);                   // Declare a predicate 'hint(X).'
			Prover<?> prover = getProver(query, new Unifier()); // Try to prove 'hint(X)'.
			Unifier result = prover.call();
			if(result != null) {                                // If successful, move to X.
				destination = result.get(x).toString();
				action = Path.path(location, destination, visited);
				location = Path.getNewLocation(location, action);
			}
		}
		// Print the action to the console before returning it.
		System.out.println(action);
		return action;
	}
	
	@Override
	public void observe(Result result) {
		// Print the result to the console.
		System.out.println(result);
		// If the player has entered a previous unvisited square...
		if(!visited.contains(location)) {
			// Add the location to the list of visited squares.
			visited.add(location);
			// Add the fact that this square has been visited to the knowledge base.
			kb.addFact(new Fact("visited", new Constant(location)));
			// If the square has a breeze, add that information to the knowledge base.
			if(result.breeze)
				kb.addFact(new Fact("breeze", new Constant(location)));
			// If the square has a stench, add that information to the knowledge base.
			if(result.stench)
				kb.addFact(new Fact("stench", new Constant(location)));
			// If the square glitters, record the location of the gold.
			if(result.glitter)
				gold = location;
		}
		// Wait some number of seconds before making the next move.
		try { Thread.sleep(Settings.DELAY * 1000); }
		catch(InterruptedException ex){/* do nothing */}
	}
	
	/**
	 * For a given expression, this method returns the appropriate
	 * {@link Prover}.
	 * 
	 * @param expression the expression to be proved
	 * @param unifier the current unifier
	 * @return the {@link Prover} object for that kind of expression
	 */
	private Prover<?> getProver(Expression expression, Unifier unifier) {
		if(expression instanceof Fact)
			return new FactProver((Fact) expression, unifier);
		else if(expression instanceof Negation)
			return new NegationProver((Negation) expression, unifier);
		else if(expression instanceof Disjunction)
			return new DisjunctionProver((Disjunction) expression, unifier);
		else if(expression instanceof Conjunction)
			return new ConjunctionProver((Conjunction) expression, unifier);
		else
			throw new IllegalArgumentException("Cannot prove " + expression.getClass().getSimpleName());
	}
	
	/**
	 * Used to prove a fact.
	 * 
	 * @author Your Name
	 */
	private class FactProver extends Prover<Fact> {

		/**
		 * Construct a prover for a given fact with the current unifier.
		 * 
		 * @param expression the fact to the proven
		 * @param unifier the current unifier
		 */
		public FactProver(Fact expression, Unifier unifier) {
			super(expression, unifier);
		}

		@Override
		protected void run() {
			// There are 2 ways to prove a fact: Either it is stated in the
			// knowledge base or it can be proven using a rule.
			// Iterate through all of the facts in KnowledgeBase#facts.
			for(Fact fact : kb.facts) {
				// If the fact unifies (see Formula#unify(Formula f, Unifier u))
				// with the expression we are trying to prove, yield the result.
				Unifier result = fact.unify(expression, unifier);
				if(result != null)
					yield(result);
			}
			// Iterate through all the rules in KnowledgeBase#rules.
			for(Rule rule : kb.rules) {
				// If the consequent of the rule (Rule#consequent) unifies with the
				// expression we are trying to prove, try to prove the antecedent
				// of the rule.
				// Do this by using the #getProver method to get the appropriate
				// prover for Rule#antecedent.  Then, until that prover is done
				// (see Prover#done), call it and yield the result.
				Unifier result = rule.consequent.unify(expression, unifier);
				if(result != null) {
					Prover<?> prover = getProver(rule.antecedent, result);
					while(!prover.done())
						yield(prover.call());
				}
			}
		}
	}
	
	/**
	 * Used to prove a negation.
	 * 
	 * @author Your Name
	 */
	private class NegationProver extends Prover<Negation> {

		/**
		 * Construct a prover for a given negation with the current unifier.
		 * 
		 * @param expression the negation to the proven
		 * @param unifier the current unifier
		 */
		public NegationProver(Negation expression, Unifier unifier) {
			super(expression, unifier);
		}

		@Override
		protected void run() {
			// Use #getProver to get the appropriate prover for the negation's
			// argument (Negation#argument).  Call the prover and see if it
			// returns a valid unifier.  If so, the argument can be proven,
			// which means the negation is false.  We only need to call this
			// prover once, because we won't care how many ways it can be true,
			// only that there exists one way that it can be true.
			Prover<?> prover = getProver(expression.argument, unifier);
			if(prover.call() == null)
				yield(unifier);
		}
	}
	
	/**
	 * Used to prove a disjunction.
	 * 
	 * @author Your Name
	 */
	private class DisjunctionProver extends Prover<Disjunction> {

		/**
		 * Construct a prover for a given disjunction with the current unifier.
		 * 
		 * @param expression the disjunction to the proven
		 * @param unifier the current unifier
		 */
		public DisjunctionProver(Disjunction expression, Unifier unifier) {
			super(expression, unifier);
		}

		@Override
		protected void run() {
			// A disjunction is true if any of its disjuncts are true.
			// Iterate through each disjunct (Disjunction#arguments).
			for(Expression argument : expression.arguments) {
				// Use #getProver to get the appropriate prover for the disjunct.
				Prover<?> prover = getProver(argument, unifier);
				// Until the prover is done (Prover#done), keep calling it and
				// yielding the result.
				while(!prover.done()) {
					Unifier result = prover.call();
					if(result != null)
						yield(result);
				}
			}
		}
	}
	
	/**
	 * Used to prove a conjunction.
	 * 
	 * @author Your Name
	 */
	private class ConjunctionProver extends Prover<Conjunction> {

		/**
		 * Construct a prover for a given conjunction with the current unifier.
		 * 
		 * @param expression the conjunction to the proven
		 * @param unifier the current unifier
		 */
		public ConjunctionProver(Conjunction expression, Unifier unifier) {
			super(expression, unifier);
		}

		@Override
		protected void run() {
			// A conjunction is true if all of its conjuncts are true.
			// The problem is that we need to check every possible way that all
			// conjuncts can be true.  We'll do this using a recursive method
			// called #proveConjunct.  State the recursive process by calling
			// that method with the first conjunct (index 0) and the current
			// unifier.
			proveConjunct(0, unifier);
		}
		
		/**
		 * Recursively checks every possible way that every conjunct can be
		 * true.
		 * 
		 * @param conjunct the index of the current conjunct to check
		 * @param u the current unifier
		 */
		private void proveConjunct(int conjunct, Unifier u) {
			// Base case: When 'conjunct' is so high that it no longer points
			// to one of the expression's conjuncts (Conjunction#arguments),
			// simply yield the current unifier.  In other words, once our
			// counter variable called 'conjunct' has increased to
			// expression.arguments.length, we know that all conjuncts have
			// been proven true.  So just yield the resulting unifier.
			if(conjunct == expression.arguments.length)
				yield(u);
			// Recursive case: Use #getProver to get the appropriate prover for
			// for the current conjunct (i.e. expression.arguments[conjunct]).
			else {
				Prover<?> prover = getProver(expression.arguments[conjunct], u);
				// Until the prover is done, call it.  If it returns a unifier, the
				// current conjunct can be proven true.  Now recursively call this
				// method with coujunct + 1 and the unifier we get got.
				while(!prover.done()) {
					Unifier result = prover.call();
					if(result != null)
						proveConjunct(conjunct + 1, result);
				}
			}
		}
	}
}
